<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\PageView;
use App\Models\NewsPost;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;

class AnalyticsController extends Controller
{
    public function index()
    {
        $totalViews   = PageView::count();
        $todayViews   = PageView::whereDate('created_at', Carbon::today())->count();
        $uniqueVisitors = PageView::distinct('session_id')->count('session_id');
        $todayUnique  = PageView::whereDate('created_at', Carbon::today())->distinct('session_id')->count('session_id');

        // Views per day for the last 30 days
        $chartData = PageView::where('created_at', '>=', Carbon::now()->subDays(30))
            ->select(DB::raw('DATE(created_at) as date'), DB::raw('COUNT(*) as views'), DB::raw('COUNT(DISTINCT session_id) as unique_visitors'))
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        // Fill missing days with 0
        $dates = [];
        $views = [];
        $uniques = [];
        for ($i = 29; $i >= 0; $i--) {
            $d = Carbon::now()->subDays($i)->format('Y-m-d');
            $dates[] = Carbon::parse($d)->format('d M');
            $found = $chartData->firstWhere('date', $d);
            $views[] = $found ? $found->views : 0;
            $uniques[] = $found ? $found->unique_visitors : 0;
        }

        // Top 10 most viewed news (by page_views table)
        $topNews = PageView::whereNotNull('news_post_id')
            ->select('news_post_id', DB::raw('COUNT(*) as view_count'))
            ->groupBy('news_post_id')
            ->orderByDesc('view_count')
            ->limit(10)
            ->with('newsPost:id,title,slug')
            ->get();

        // Top referrers
        $topReferrers = PageView::whereNotNull('referer')
            ->where('referer', '!=', '')
            ->select('referer', DB::raw('COUNT(*) as count'))
            ->groupBy('referer')
            ->orderByDesc('count')
            ->limit(10)
            ->get();

        // Top pages
        $topPages = PageView::select('url', DB::raw('COUNT(*) as count'))
            ->groupBy('url')
            ->orderByDesc('count')
            ->limit(10)
            ->get();

        return view('admin.analytics.index', compact(
            'totalViews', 'todayViews', 'uniqueVisitors', 'todayUnique',
            'dates', 'views', 'uniques',
            'topNews', 'topReferrers', 'topPages'
        ));
    }
}
